<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Lib\FormProcessor;
use App\Lib\RequiredConfig;
use App\Models\LoanPlan;
use App\Models\TimeInterval;
use Illuminate\Http\Request;

class LoanPlanController extends Controller {
    public function index() {
        $pageTitle = 'Loan Plans';
        $loanPlans = LoanPlan::searchable(['name'])->orderByDesc('id')->paginate(getPaginate());
        $days      = TimeInterval::select('name', 'day')->get()->toArray();
        return view('admin.plan.loan', compact('pageTitle', 'loanPlans', 'days'));
    }

    public function create() {
        $pageTitle = 'Add New Loan Plan';
        $days      = TimeInterval::all();
        return view('admin.plan.loan_form', compact('pageTitle', 'days'));
    }

    public function edit($id) {
        $pageTitle = 'Edit Loan Plan';
        $plan      = LoanPlan::findOrFail($id);
        $days      = TimeInterval::all();
        $form      = $plan->form;
        return view('admin.plan.loan_form', compact('pageTitle', 'plan', 'days', 'form'));
    }

    private function validation($request, $formProcessor) {
        $validation = [
            'name'                 => 'required',
            'installment_interval' => 'required|integer|exists:time_intervals,day',
            'loan_amount'          => 'required|numeric|gt:0|lt:payable_amount',
            'payable_amount'       => 'required|numeric|gt:loan_amount',
            'total_installment'    => 'required|integer|min:1',
            'description'          => 'required',
            'fixed_late_fee'       => 'required|numeric|min:0',
            'percent_late_fee'     => 'required|numeric|min:0',
            'field_name.*'         => 'sometimes|required',
        ];

        $generatorValidation = $formProcessor->generatorValidation();
        $validation          = array_merge($validation, $generatorValidation['rules']);
        $request->validate($validation, $generatorValidation['messages']);
    }

    public function store(Request $request, $id = 0) {
        $formProcessor = new FormProcessor();
        $this->validation($request, $formProcessor);
        if ($id) {
            $loanPlan     = LoanPlan::findOrFail($id);
            $generate     = $formProcessor->generate('loan_form', true, 'id', $loanPlan->form_id);
            $notification = "Loan plan updated successfully";
        } else {
            $generate     = $formProcessor->generate('loan_form');
            $loanPlan     = new LoanPlan();
            $notification = "Loan plan created successfully";
        }
        $loanPlan->form_id              = @$generate->id ?? 0;
        $loanPlan->name                 = $request->name;
        $loanPlan->loan_amount          = $request->loan_amount;
        $loanPlan->payable_amount       = $request->payable_amount;
        $loanPlan->installment          = $request->payable_amount / $request->total_installment;
        $loanPlan->total_installment    = $request->total_installment;
        $loanPlan->installment_interval = $request->installment_interval;
        $loanPlan->fixed_late_fee       = $request->fixed_late_fee;
        $loanPlan->percent_late_fee     = $request->percent_late_fee;
        $loanPlan->description          = $request->description;
        $loanPlan->save();

        RequiredConfig::configured('loan_plan');
        $notify[] = ['success', $notification];
        return back()->withNotify($notify);
    }

    public function status($id) {
        return LoanPlan::changeStatus($id);
    }
}
