<?php

namespace App\Http\Controllers\Admin;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Models\PaidLog;
use App\Models\TimeInterval;
use Carbon\Carbon;
use Illuminate\Http\Request;

class ManageCollectionController extends Controller {
    public function index($date = null, $staffId = null, $adminReceive = null) {
        $segment     = request()->segment(3);
        $collections = PaidLog::query();

        switch ($segment) {
        case 'loan':
            $pageTitle = 'Loan Collections';
            $collections->where('loan_id', '!=', 0)->with('loan.loanPlan');
            break;
        case 'savings':
            $pageTitle = 'Savings Collections';
            $collections->where('savings_id', '!=', 0)->with('savings.savingsPlan');
            break;
        default:
            $pageTitle = 'All Collections';
            $collections->with('loan.loanPlan', 'savings.savingsPlan');
            if ($date == 'yesterday') {
                $collections->whereDate('created_at', Carbon::yesterday()->format('Y-m-d'));
            }
            break;
        }

        if ($date == 'today') {
            $collections->whereDate('created_at', Carbon::now()->format('Y-m-d'));
        } elseif($date) {
            $collections->whereDate('created_at', $date);
        }

        if ($staffId) {
            $collections->where('staff_id', $staffId)->whereDate('created_at', $date);
        }

        if ($adminReceive !== null) {
            $collections->where('admin_receive', $adminReceive);
        }

        $collections = $collections->searchable(['user:username,lastname,firstname', 'staff:username,lastname,firstname'])
            ->success()
            ->with('user', 'staff')
            ->orderByDesc('id')
            ->paginate(getPaginate());

        $days = TimeInterval::select('name', 'day')->get()->toArray();

        return view('admin.collection.index', compact('pageTitle', 'collections', 'days'));
    }

    public function staffCollection() {
        $segmentThree = request()->segment(3);
        $segmentFour  = request()->segment(5);

        $collections = PaidLog::where('staff_id', '!=', 0);
        $pageTitle   = 'Collections By Staff';

        switch ($segmentThree) {
        case 'loan':
            $pageTitle = 'Loan Collections By Staff';
            $collections->where('loan_id', '!=', 0);
            break;
        case 'savings':
            $pageTitle = 'Savings Collections By Staff';
            $collections->where('savings_id', '!=', 0);
            break;
        }

        $collections->groupBy('staff_id')
            ->groupByRaw("DATE_FORMAT(created_at, '%Y-%m-%d')")
            ->selectRaw('staff_id, loan_id, DATE_FORMAT(created_at, "%Y-%m-%d") as date, sum(amount) as total_amount, count(*) as count');

        switch ($segmentFour) {
        case 'pending':
            $pageTitle = 'Pending ' . $pageTitle;
            $collections->where('admin_receive', Status::NO);
            break;
        case 'paid':
            $pageTitle = 'Paid ' . $pageTitle;
            $collections->where('admin_receive', Status::YES);
            break;
        }

        $collections = $collections->searchable(['user:username,lastname,firstname', 'staff:username,lastname,firstname'])
            ->success()->with('staff')->latest()->paginate(getPaginate());

        return view('admin.collection.by_staff', compact('pageTitle', 'collections'));
    }

    public function confirmation(Request $request) {
        $request->validate([
            'staff_id' => 'required',
            'date'     => 'required|date',
        ]);
        $paidLogs = PaidLog::where('staff_id', $request->staff_id)
            ->whereDate('created_at', Carbon::parse($request->date));

        if ($request->loan_id) {
            $paidLogs = $paidLogs->where('loan_id', '!=', 0);
        } else {
            $paidLogs = $paidLogs->where('savings_id', '!=', 0);
        }

        $paidLogs->update(['admin_receive' => Status::YES]);
        $notify[] = ['success', 'Received collections successfully'];
        return back()->withNotify($notify);
    }
}
