<?php

namespace App\Http\Controllers\Admin;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\LoanProcess;
use App\Models\Loan;
use App\Models\LoanPlan;
use App\Models\PaidLog;
use App\Models\TimeInterval;
use App\Models\User;
use Illuminate\Http\Request;

class ManageLoanController extends Controller {

    protected $pageTitle = 'All Loans';
    protected $view      = 'admin.loan.index';
    protected $loans     = null;

    public function index() {
        $data = $this->loanData();
        return view($this->view, $data);
    }

    public function paid() {
        $data = $this->loanData('paid');
        return view($this->view, $data);
    }

    public function active() {
        $data = $this->loanData('active');
        return view($this->view, $data);
    }

    public function pending() {
        $data = $this->loanData('pending');
        return view($this->view, $data);
    }

    protected function loanData($scope = null) {
        if ($scope) {
            $this->pageTitle = ucfirst($scope) . ' Loans';
            $loans           = Loan::$scope();
        } else {
            $loans = Loan::query();
        }

        $loans = $loans->searchable(['user:username', 'loanPlan:name'])
            ->with(['user', 'loanPlan', 'paidLogs' => function ($paidLog) {
                $paidLog->success();
            }])
            ->orderBy('id', 'desc')
            ->paginate(getPaginate());

        $data['loans']     = $loans;
        $data['days']      = TimeInterval::select('name', 'day')->get()->toArray();
        $data['pageTitle'] = $this->pageTitle;
        return $data;
    }

    public function register(Request $request) {
        $pageTitle = 'Register New Loan';
        $plans     = LoanPlan::active()->get();
        $days      = TimeInterval::select('name', 'day')->get()->toArray();
        return view('admin.loan.register', compact('pageTitle', 'plans', 'days'));
    }

    public function storeRegister(Request $request) {
        $request->validate([
            'user'    => 'required',
            'plan_id' => 'required',
        ], [
            'user.required' => 'Username or email field required',
        ]);

        $user = User::where('email', $request->user)->orWhere('username', $request->user)->firstOrFail();
        $plan = LoanPlan::findOrFail($request->plan_id);

        $loan                       = new Loan();
        $loan->user_id              = $user->id;
        $loan->loan_plan_id         = $plan->id;
        $loan->loan_amount          = $plan->loan_amount;
        $loan->payable_amount       = $plan->payable_amount;
        $loan->installment          = $plan->installment;
        $loan->installment_interval = $plan->installment_interval;
        $loan->total_installment    = $plan->total_installment;
        $loan->next_installment     = now()->addDays($plan->installment_interval);
        $loan->late_fee             = $plan->fixed_late_fee + ($plan->installment * $plan->percent_late_fee / 100);
        $loan->status               = Status::LOAN_ACTIVE;
        $loan->save();

        notify($loan->user, 'LOAN_REGISTER', [
            'loan_plan'            => $loan->loanPlan->name,
            'amount'               => showAmount($loan->loan_amount),
            'currency'             => gs('cur_text'),
            'installment'          => showAmount($loan->installment),
            'total_installment'    => $loan->total_installment,
            'installment_interval' => $loan->installment_interval . ' days',
        ]);

        $notify[] = ['success', 'Loan registered/given successfully'];
        return back()->withNotify($notify);
    }

    public function details($id) {
        $pageTitle = 'Pending Loan Details';
        $loan      = Loan::with('loanPlan')->findOrFail($id);
        $days      = TimeInterval::select('name', 'day')->get()->toArray();
        return view('admin.loan.details', compact('pageTitle', 'loan', 'days'));
    }

    public function approve($id) {
        $loan                   = Loan::with('user', 'loanPlan')->findOrFail($id);
        $loan->next_installment = now()->addDays($loan->installment_interval);
        $loan->status           = Status::LOAN_ACTIVE;
        $loan->save();

        notify($loan->user, 'LOAN_REGISTER', [
            'loan_plan'            => $loan->loanPlan->name,
            'amount'               => showAmount($loan->loan_amount),
            'currency'             => gs('cur_text'),
            'installment'          => showAmount($loan->installment),
            'total_installment'    => $loan->total_installment,
            'installment_interval' => $loan->installment_interval . ' days',
        ]);

        $notify[] = ['success', 'Loan approved successfully'];
        return back()->withNotify($notify);
    }

    public function installment(Request $request) {
        $loan = Loan::with(['user', 'loanPlan', 'paidLogs' => function ($paidLog) {
            $paidLog->success();
        }])->findOrFail($request->loan_id);

        $user        = $loan->user;
        $loanProcess = new LoanProcess([
            'user' => $user,
        ]);

        $response = $loanProcess->installment($loan);
        if ($response['error'] == true) {
            return back()->withNotify($response['message']);
        }

        $paidLog               = new PaidLog();
        $paidLog->loan_id      = $request->loan_id;
        $paidLog->user_id      = $loan->user->id;
        $paidLog->amount       = $loan->installment;
        $paidLog->late_fee     = $loanProcess->lateFee;
        $paidLog->final_amount = $loanProcess->amountWithLateFee;
        $paidLog->status       = 1;
        $paidLog->save();

        $loan->total_paid += $loan->installment;
        $loan->installment_given += 1;
        $loan->total_late_fee_paid += $loanProcess->lateFee;
        $loan->last_installment = now();
        $loan->next_installment = $loan->next_installment->addDays($loan->installment_interval);

        if ($loan->total_installment == $loan->installment_given) {
            $loan->status = Status::LOAN_PAID;
        }
        $loan->save();

        notify($loan->user, 'LOAN_PAID', [
            'amount'    => showAmount($loanProcess->amountWithLateFee),
            'currency'  => gs('cur_text'),
            'paid_by'   => 'Admin',
            'loan_plan' => $loan->loanPlan->name,
        ]);

        $notify[] = ['success', "Installment taken successfully"];
        return back()->withNotify($notify);
    }

    public function close(Request $request) {
        $loan         = Loan::active()->findOrFail($request->loan_id);
        $loan->status = Status::LOAN_CLOSED;
        $loan->save();

        $notify[] = ['success', 'Loan closed successfully'];
        return back()->withNotify($notify);
    }

    public function userLoans($id) {
        $user      = User::findOrFail($id);
        $pageTitle = $user->fullname . ' active loans';
        $loans     = Loan::active()->where('user_id', $id)->with(['user', 'loanPlan', 'paidLogs' => function ($paidLogs) {
            $paidLogs->success();
        }])->paginate(getPaginate(20));
        $days = TimeInterval::select('name', 'day')->get()->toArray();

        return view('admin.loan.index', compact('pageTitle', 'loans', 'days'));
    }
}
