<?php

namespace App\Http\Controllers\Admin;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\SavingsProcess;
use App\Models\GeneralSetting;
use App\Models\PaidLog;
use App\Models\Savings;
use App\Models\SavingsPlan;
use App\Models\TimeInterval;
use App\Models\User;
use Illuminate\Http\Request;

class ManageSavingsController extends Controller {

    protected $savings = null;
    public function index() {
        $pageTitle = 'All Savings';
        $data      = (object) $this->savingsData();
        return view('admin.savings.index', compact('pageTitle', 'data'));
    }

    public function pendingMatured() {
        $pageTitle = 'Pending Matured Savings';
        $data      = (object) $this->savingsData('pendingMatured');
        return view('admin.savings.index', compact('pageTitle', 'data'));
    }
    public function paidMatured() {
        $pageTitle = 'Paid Matured Savings';
        $data      = (object) $this->savingsData('paidMatured');
        return view('admin.savings.index', compact('pageTitle', 'data'));
    }

    public function active() {
        $pageTitle = 'Active Savings';
        $data      = (object) $this->savingsData('active');
        return view('admin.savings.index', compact('pageTitle', 'data'));
    }

    public function pending() {
        $pageTitle = 'Pending Savings';
        $data      = (object) $this->savingsData('pending');
        return view('admin.savings.index', compact('pageTitle', 'data'));
    }

    protected function savingsData($scope = null) {
        if ($scope) {
            $savings = Savings::$scope();
        } else {
            $savings = Savings::query();
        }
        $savings = $savings->searchable(['user:username', 'savingsPlan:name'])
            ->with(['user', 'savingsPlan', 'paidLogs' => function ($paidLog) {
                $paidLog->success();
            }])->orderBy('id', 'desc')->paginate(getPaginate());

        $data['savings'] = $savings;
        $data['days']    = TimeInterval::select('name', 'day')->get()->toArray();
        return $data;
    }

    public function maturedTransfer(Request $request, $id) {
        // dd($id);
        $request->validate([
            'savings_id' => 'required',
        ]);

        $savings                = Savings::paid()->where('transfer_user', 0)->findOrFail($request->savings_id);
        $savings->transfer_user = 1;
        $savings->save();

        $general = GeneralSetting::first();
        notify($savings->user, 'MATURED_PAID', [
            'savings_plan'    => $savings->savingsPlan->name,
            'savings_amount'  => showAmount($savings->savings_amount, currencyFormat: false),
            'giveable_amount' => showAmount($savings->giveable_amount, currencyFormat: false),
            'currency'        => $general->cur_text,
        ]);

        $notify[] = ['success', 'Savings balance successfully transfer to user'];
        return back()->withNotify($notify);
    }

    public function register(Request $request) {
        $pageTitle = 'Register New Savings';
        $plans     = SavingsPlan::active()->get();
        $days      = TimeInterval::select('name', 'day')->get()->toArray();
        return view('admin.savings.register', compact('pageTitle', 'plans', 'days'));
    }

    public function storeRegister(Request $request) {
        $request->validate([
            'user'    => 'required',
            'plan_id' => 'required',
        ], [
            'user.required' => 'Username or email field required',
        ]);

        $user = User::where('email', $request->user)->orWhere('username', $request->user)->firstOrFail();
        $plan = SavingsPlan::findOrFail($request->plan_id);

        $savings                       = new Savings();
        $savings->user_id              = $user->id;
        $savings->savings_plan_id      = $plan->id;
        $savings->savings_amount       = $plan->savings_amount;
        $savings->giveable_amount      = $plan->giveable_amount;
        $savings->installment          = $plan->installment;
        $savings->installment_interval = $plan->installment_interval;
        $savings->total_installment    = $plan->total_installment;
        $savings->next_installment     = now()->addDays($plan->installment_interval);
        $savings->late_fee             = $plan->fixed_late_fee + ($plan->installment * $plan->percent_late_fee / 100);
        $savings->save();

        $general = GeneralSetting::first();
        notify($savings->user, 'SAVINGS_REGISTER', [
            'savings_plan'         => $savings->savingsPlan->name,
            'amount'               => showAmount($savings->savings_amount, currencyFormat: false),
            'currency'             => $general->cur_text,
            'installment'          => showAmount($savings->installment, currencyFormat: false),
            'total_installment'    => $savings->total_installment,
            'installment_interval' => $savings->installment_interval . ' days',
        ]);

        $notify[] = ['success', 'Savings registered/added successfully'];
        return back()->withNotify($notify);
    }

    public function details($id) {
        $pageTitle = 'Pending Savings Details';
        $savings   = Savings::with('savingsPlan')->findOrFail($id);
        $days      = TimeInterval::select('name', 'day')->get()->toArray();

        return view('admin.savings.details', compact('pageTitle', 'savings', 'days'));
    }

    public function approve($id) {
        $savings                   = Savings::with('user', 'savingsPlan')->findOrFail($id);
        $savings->next_installment = now();
        $savings->status           = Status::SAVINGS_ACTIVE;
        $savings->save();

        notify($savings->user, 'SAVINGS_REGISTER', [
            'savings_plan'         => $savings->savingsPlan->name,
            'amount'               => showAmount($savings->savings_amount, currencyFormat: false),
            'currency'             => gs('cur_text'),
            'installment'          => showAmount($savings->installment, currencyFormat: false),
            'total_installment'    => $savings->total_installment,
            'installment_interval' => $savings->installment_interval . ' days',
        ]);

        $notify[] = ['success', 'Savings approved successfully'];
        return back()->withNotify($notify);
    }

    public function installment(Request $request) {
        $savings = Savings::with(['user', 'savingsPlan', 'paidLogs' => function ($paidLog) {
            $paidLog->success();
        }])->findOrFail($request->savings_id);

        $user           = $savings->user;
        $savingsProcess = new SavingsProcess([
            'user' => $user,
        ]);

        $response = $savingsProcess->installment($savings);
        if ($response['error'] == true) {
            return back()->withNotify($response['message']);
        }

        $paidLog               = new PaidLog();
        $paidLog->savings_id   = $request->savings_id;
        $paidLog->user_id      = $savings->user->id;
        $paidLog->amount       = $savingsProcess->installment;
        $paidLog->late_fee     = $savingsProcess->lateFee;
        $paidLog->final_amount = $savingsProcess->amountWithLateFee;
        $paidLog->status       = 1;
        $paidLog->save();

        $savingsProcess->updateSavings($savings);

        $general = GeneralSetting::first();
        notify($savings->user, 'SAVINGS_PAID', [
            'amount'       => showAmount($savingsProcess->amountWithLateFee, currencyFormat: false),
            'currency'     => $general->cur_text,
            'paid_by'      => 'Admin',
            'savings_plan' => $savings->savingsPlan->name,
        ]);

        $notify[] = ['success', "Installment taken successfully"];
        return back()->withNotify($notify);
    }

    public function close(Request $request) {
        $savings         = Savings::active()->findOrFail($request->savings_id);
        $savings->status = Status::LOAN_CLOSED;
        $savings->save();

        $notify[] = ['success', 'Savings close successfully'];
        return back()->withNotify($notify);
    }

    public function userSavings($id) {
        $user      = User::findOrFail($id);
        $pageTitle = $user->fullname . ' active savings';

        $savingsList = Savings::active()->where('user_id', $id)->with(['user', 'savingsPlan', 'paidLogs' => function ($paidLogs) {
            $paidLogs->success();
        }])->paginate(getPaginate(20));
        $days = TimeInterval::select('name', 'day')->get()->toArray();

        $savingData['savings'] = $savingsList;
        $savingData['days']    = $days;

        $data = (object) $savingData;

        return view('admin.savings.index', compact('pageTitle', 'savingsList', 'days', 'data'));
    }
}
