<?php

namespace App\Http\Controllers\Staff;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\LoanProcess;
use App\Models\Loan;
use App\Models\LoanPlan;
use App\Models\PaidLog;
use App\Models\TimeInterval;
use App\Models\User;
use Illuminate\Http\Request;

class LoanController extends Controller {
    public function index() {
        $segments    = request()->segments();
        $lastSegment = end($segments);

        switch ($lastSegment) {
        case 'pending':
            $pageTitle = 'Pending Loans';
            $loans     = Loan::pending()->where('staff_id', auth('staff')->id());
            break;
        case 'active':
            $pageTitle = 'Active Loans';
            $loans     = Loan::active();
            break;
        case 'paid':
            $pageTitle = 'Paid Loans';
            $loans     = Loan::paid();
            break;
        case 'closed':
            $pageTitle = 'Closed Loans';
            $loans     = Loan::closed();
            break;
        default:
            $pageTitle = 'All Loans';
            $loans     = Loan::query();
            break;
        }

        $loans = $loans->searchable(['user:username', 'loanPlan:name'])
            ->with(['user', 'loanPlan', 'paidLogs' => function ($paidLog) {
                $paidLog->success();
            }])
            ->latest()
            ->paginate(getPaginate());

        $days = TimeInterval::select('name', 'day')->get()->toArray();

        return view('staff.loan.index', compact('pageTitle', 'loans', 'days'));
    }

    public function plan() {
        $pageTitle = 'Loan Plans';
        $loanPlans = LoanPlan::active()->latest()->paginate(getPaginate());
        $days      = TimeInterval::select('name', 'day')->get()->toArray();

        return view('staff.loan.plan', compact('pageTitle', 'loanPlans', 'days'));
    }

    public function applyForm($id) {
        $pageTitle = 'Apply For Loan';
        $plan      = LoanPlan::active()->findOrFail($id);
        $days      = TimeInterval::select('name', 'day')->get()->toArray();
        return view('staff.loan.apply', compact('pageTitle', 'plan', 'days'));
    }

    public function apply(Request $request, $id) {
        $plan        = LoanPlan::active()->findOrFail($id);
        $user        = User::where('username', $request->user)->orWhere('email', $request->user)->firstOrFail();
        $loanProcess = new LoanProcess(['plan' => $plan, 'user' => $user, 'staff_id' => auth('staff')->id()]);
        $loanProcess->apply();
        $notify[] = ['success', 'You have successfully applied for the loan. Please wait for authority approval.'];
        return to_route('staff.loan.pending')->withNotify($notify);
    }

    public function installment(Request $request) {
        $request->validate([
            'loan_id' => 'required',
        ]);

        $loan = Loan::where('id', $request->loan_id)->with(['user', 'loanPlan', 'paidLogs' => function ($paidLog) {
            $paidLog->success();
        }])->firstOrFail();

        $trx         = getTrx();
        $user        = $loan->user;
        $loanProcess = new LoanProcess(['user' => $user]);

        $response = $loanProcess->installment($loan);
        if ($response['error'] == true) {
            return back()->withNotify($response['notify']);
        }

        $paidLog               = new PaidLog();
        $paidLog->loan_id      = $loan->id;
        $paidLog->user_id      = $user->id;
        $paidLog->amount       = $loan->installment;
        $paidLog->late_fee     = $loanProcess->lateFee;
        $paidLog->final_amount = $loanProcess->amountWithLateFee;
        $paidLog->trx          = $trx;
        $paidLog->save();

        $paidLog                = new PaidLog();
        $paidLog->loan_id       = $loan->id;
        $paidLog->user_id       = $user->id;
        $paidLog->staff_id      = auth('staff')->id();
        $paidLog->admin_receive = Status::NO;
        $paidLog->amount        = $loan->installment;
        $paidLog->late_fee      = $loanProcess->lateFee;
        $paidLog->final_amount  = $loanProcess->amountWithLateFee;
        $paidLog->trx           = $trx;
        $paidLog->status        = Status::PAID_SUCCESS;
        $paidLog->save();

        $loanProcess->updateLoan($loan);

        notify($user, 'LOAN_INSTALLMENT_PAID', [
            'amount'    => showAmount($loanProcess->amountWithLateFee, currencyFormat: false),
            'currency'  => gs('cur_text'),
            'paid_by'   => 'Staff',
            'loan_plan' => $loan->loanPlan->name,
        ]);

        $notify[] = ['success', "Loan installment taken successfully"];
        return back()->withNotify($notify);
    }
}
