<?php

namespace App\Http\Controllers\Staff;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\SavingsProcess;
use App\Models\PaidLog;
use App\Models\Savings;
use App\Models\SavingsPlan;
use App\Models\TimeInterval;
use App\Models\User;
use Illuminate\Http\Request;

class SavingsController extends Controller {

    public function index() {
        $segments    = request()->segments();
        $lastSegment = end($segments);

        switch ($lastSegment) {
        case 'pending':
            $pageTitle   = 'Pending Savings';
            $savingsList = Savings::pending()->where('staff_id', auth('staff')->id());
            break;
        case 'active':
            $pageTitle   = 'Active Savings';
            $savingsList = Savings::active();
            break;
        case 'paid':
            $pageTitle   = 'Paid Savings';
            $savingsList = Savings::paid();
            break;
        case 'closed':
            $pageTitle   = 'Closed Savings';
            $savingsList = Savings::closed();
            break;
        default:
            $pageTitle   = 'All Savings';
            $savingsList = Savings::query();
            break;
        }
        $savingsList = $savingsList->searchable(['user:username', 'loanPlan:name'])
            ->with(['user', 'savingsPlan', 'paidLogs' => function ($paidLog) {
                $paidLog->success();
            }])
            ->latest()
            ->paginate(getPaginate());

        $days = TimeInterval::select('name', 'day')->get()->toArray();
        return view('staff.savings.index', compact('pageTitle', 'savingsList', 'days'));
    }

    public function plan() {
        $pageTitle    = 'Savings Plans';
        $savingsPlans = SavingsPlan::active()->paginate(getPaginate());
        $days         = TimeInterval::select('name', 'day')->get()->toArray();
        return view('staff.savings.plan', compact('pageTitle', 'savingsPlans', 'days'));
    }

    public function applyForm($planId) {
        $pageTitle = 'Apply For Savings';
        $plan      = SavingsPlan::active()->findOrFail($planId);
        $days      = TimeInterval::select('name', 'day')->get()->toArray();
        return view('staff.savings.apply', compact('pageTitle', 'plan', 'days'));
    }

    public function apply(Request $request, $id) {
        $plan        = SavingsPlan::active()->findOrFail($id);
        $user        = User::where('username', $request->user)->orWhere('email', $request->user)->firstOrFail();
        $loanProcess = new SavingsProcess(['plan' => $plan, 'user' => $user, 'staff_id' => auth('staff')->id()]);
        $loanProcess->apply();
        $notify[] = ['success', 'You have successfully applied for the Savings.'];
        return to_route('staff.savings.pending')->withNotify($notify);
    }

    public function installment(Request $request) {
        $request->validate([
            'savings_id' => 'required',
        ]);

        $savings = Savings::with(['user', 'savingsPlan', 'paidLogs' => function ($paidLog) {
            $paidLog->success();
        }])->findOrFail($request->savings_id);

        $trx            = getTrx();
        $user           = $savings->user;
        $savingsProcess = new SavingsProcess([
            'user' => $user,
        ]);

        $response = $savingsProcess->installment($savings);
        if ($response['error'] == true) {
            return back()->withNotify($response['notify']);
        }

        $paidLog                = new PaidLog();
        $paidLog->savings_id    = $savings->id;
        $paidLog->user_id       = $user->id;
        $paidLog->staff_id      = auth('staff')->id();
        $paidLog->admin_receive = Status::NO;
        $paidLog->amount        = $savings->installment;
        $paidLog->late_fee      = $savingsProcess->lateFee;
        $paidLog->final_amount  = $savingsProcess->amountWithLateFee;
        $paidLog->trx           = $trx;
        $paidLog->status        = Status::PAID_SUCCESS;
        $paidLog->save();

        $savingsProcess->updateSavings($savings);

        notify($savings->user, 'SAVINGS_INSTALLMENT_PAID', [
            'amount'       => showAmount($savingsProcess->amountWithLateFee, currencyFormat: false),
            'currency'     => gs('cur_text'),
            'paid_by'      => 'Staff',
            'savings_plan' => $savings->savingsPlan->name,
        ]);

        $notify[] = ['success', "Savings Installment taken successfully"];
        return back()->withNotify($notify);
    }
}
