<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Lib\LoanProcess;
use App\Models\TimeInterval;
use App\Models\Loan;
use App\Models\LoanPlan;
use App\Models\PaidLog;
use Illuminate\Http\Request;


class LoanController extends Controller
{
    public function index()
    {
        $segments = request()->segments();
        $lastSegment = end($segments);

        switch ($lastSegment) {
            case 'pending':
                $pageTitle = 'Pending Loans';
                $loans = Loan::pending();
                break;
            case 'active':
                $pageTitle = 'Active Loans';
                $loans = Loan::active();
                break;
            case 'paid':
                $pageTitle = 'Paid Loans';
                $loans = Loan::paid();
                break;
            case 'closed':
                $pageTitle = 'Closed Loans';
                $loans = Loan::closed();
                break;
            default:
                $pageTitle = 'All Loans';
                $loans = Loan::query();
                break;
        }

        $loans = $loans->where('user_id', auth()->id())
            ->with('loanPlan')
            ->latest()
            ->paginate(getPaginate());

        $days = TimeInterval::select('name', 'day')->get()->toArray();

        return view('user.loan.index', compact('pageTitle', 'loans', 'days'));
    }

    public function plan()
    {
        $pageTitle = 'Loan Plans';
        $loanPlans = LoanPlan::active()->latest()->paginate(getPaginate());
        $days      = TimeInterval::select('name', 'day')->get()->toArray();

        return view('user.loan.plan', compact('pageTitle', 'loanPlans', 'days'));
    }

    public function applyForm($id)
    {
        $pageTitle = 'Apply For Loan';
        $plan      = LoanPlan::active()->findOrFail($id);
        $days      = TimeInterval::select('name', 'day')->get()->toArray();
        return view('user.loan.apply', compact('pageTitle', 'plan', 'days'));
    }

    public function apply($id)
    {
        $plan = LoanPlan::active()->findOrFail($id);
        $loanProcess = new LoanProcess(['plan' => $plan, 'user' => auth()->user()]);
        $loanProcess->apply();
        $notify[] = ['success', 'You have successfully applied for the loan. Please wait for authority approval.'];
        return to_route('user.loan.pending')->withNotify($notify);
    }

    public function payment(Request $request)
    {
        $request->validate([
            'loan_id' => 'required',
        ]);

        $loan = Loan::where('id', $request->loan_id)->where('user_id', auth()->id())->with('loanPlan')->firstOrFail();

        $trx  = getTrx();
        $user = auth()->user();
        $loanProcess = new LoanProcess(['user' => $user]);

        $response = $loanProcess->installment($loan);
        if ($response['error'] == true) {
            return back()->withNotify($response['notify']);
        }

        $paidLog               = new PaidLog();
        $paidLog->loan_id      = $loan->id;
        $paidLog->user_id      = $user->id;
        $paidLog->amount       = $loan->installment;
        $paidLog->late_fee     = $loanProcess->lateFee;
        $paidLog->final_amount = $loanProcess->amountWithLateFee;
        $paidLog->trx          = $trx;
        $paidLog->save();

        session()->put('payment_data', [
            'paid_log_id' => $paidLog->id,
            'amount'      => $loanProcess->amountWithLateFee,
            'trx'         => $paidLog->trx
        ]);

        return to_route('user.deposit.index');
    }
}
