<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Lib\SavingsProcess;
use App\Models\TimeInterval;
use App\Models\PaidLog;
use App\Models\Savings;
use App\Models\SavingsPlan;
use Illuminate\Http\Request;

class SavingsController extends Controller
{

    public function index()
    {
        $segments = request()->segments();
        $lastSegment = end($segments);
    
        switch ($lastSegment) {
            case 'pending':
                $pageTitle = 'Pending Savings';
                $savingsList = Savings::pending();
                break;
            case 'active':
                $pageTitle = 'Active Savings';
                $savingsList = Savings::active();
                break;
            case 'paid':
                $pageTitle = 'Paid Savings';
                $savingsList = Savings::paid();
                break;
            case 'closed':
                $pageTitle = 'Closed Savings';
                $savingsList = Savings::closed();
                break;
            default:
                $pageTitle = 'All Savings';
                $savingsList = Savings::query();
                break;
        }
        $savingsList = $savingsList->where('user_id', auth()->id())
                                   ->with('savingsPlan')
                                   ->latest()
                                   ->paginate(getPaginate());
    
        $days = TimeInterval::select('name', 'day')->get()->toArray();
        return view('user.savings.index', compact('pageTitle', 'savingsList', 'days'));
    }
    

    public function plan(){
        $pageTitle = 'Savings Plans';
        $savingsPlans = SavingsPlan::active()->paginate(getPaginate());
        $days = TimeInterval::select('name', 'day')->get()->toArray();
        return view('user.savings.plan', compact('pageTitle','savingsPlans', 'days'));
    }

    public function applyForm($planId){
        $pageTitle = 'Apply For Savings';
        $plan = SavingsPlan::active()->findOrFail($planId);
        $days = TimeInterval::select('name', 'day')->get()->toArray();
        return view('user.savings.apply', compact('pageTitle', 'plan', 'days'));
    }

    public function apply($id)
    {
        $plan = SavingsPlan::active()->findOrFail($id);
        $loanProcess = new SavingsProcess(['plan' => $plan, 'user' => auth()->user()]);
        $loanProcess->apply();
        $notify[] = ['success', 'You have successfully applied for the Savings.'];
        return to_route('user.savings.pending')->withNotify($notify);
    }
   
    public function payment(Request $request){
        $request->validate([
            'savings_id' => 'required',
        ]);

        $savings = Savings::where('id', $request->savings_id)->where('user_id', auth()->id())->with('savingsPlan')->firstOrFail();

        $trx = getTrx();
        $user =  auth()->user();
        $savingsProcess = new SavingsProcess([
            'user'=>$user,
        ]);
    
        $response = $savingsProcess->installment($savings);
        if ($response['error'] == true) {
            return back()->withNotify($response['notify']);
        }

        $paidLog               = new PaidLog();
        $paidLog->savings_id   = $savings->id;
        $paidLog->user_id      = auth()->id();
        $paidLog->amount       = $savings->installment;
        $paidLog->late_fee     = $savingsProcess->lateFee;
        $paidLog->final_amount = $savingsProcess->amountWithLateFee;
        $paidLog->trx          = $trx;

        $paidLog->save();
        session()->put('payment_data',[
            'paid_log_id' => $paidLog->id,
            'amount'  => $savingsProcess->amountWithLateFee,
            'trx'         => $paidLog->trx
        ]);
        return to_route('user.deposit.index');
    }
}
